-- RMC_BeltAnchor.lua (v1.1.0.0)
-- Adds mass scaling while strapped to reduce suspension squat. Restores on unstrap.
RMCBeltAnchor = {}
RMCBeltAnchor.DEBUG = false
RMCBeltAnchor.MODE = "MULTI_JOINT" -- or "KINEMATIC_LOCK"
RMCBeltAnchor.MASS_SCALE = 0.85

RMCBeltAnchor.OFFSETS = { {0,0,0}, {0.10,0,0}, {0,0,0.10} }
RMCBeltAnchor.SPRING_K = 100000
RMCBeltAnchor.SPRING_D = 150

local function _consoleSetMode(self, mode)
    if not mode or mode == "" then
if RMCBeltAnchor.DEBUG then print(("[RMC_BeltAnchor] Mode = %s"):format(RMCBeltAnchor.MODE)) end
        return
    end
    local up = tostring(mode):upper()
    if up == "MULTI_JOINT" or up == "KINEMATIC_LOCK" then
        RMCBeltAnchor.MODE = up
if RMCBeltAnchor.DEBUG then print(("[RMC_BeltAnchor] Set mode = %s"):format(RMCBeltAnchor.MODE)) end
    else
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Invalid mode. Use MULTI_JOINT or KINEMATIC_LOCK") end
    end
end
addConsoleCommand("gsRMCBeltAnchorMode", "Get/Set RMC Belt Anchor mode (MULTI_JOINT|KINEMATIC_LOCK)", "_consoleSetMode", RMCBeltAnchor)

local function _consoleSetMassScale(self, scale)
    if scale == nil or scale == "" then
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] MASS_SCALE = %.3f", RMCBeltAnchor.MASS_SCALE)) end
        return
    end
    local v = tonumber(scale)
    if v and v > 0.1 and v <= 1.5 then
        RMCBeltAnchor.MASS_SCALE = v
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Set MASS_SCALE = %.3f", RMCBeltAnchor.MASS_SCALE)) end
    else
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Invalid MASS_SCALE. Range 0.1 .. 1.5") end
    end
end
addConsoleCommand("gsRMCBeltAnchorMassScale", "Get/Set mass scale while strapped (e.g. 0.85).", "_consoleSetMassScale", RMCBeltAnchor)

local function _hardReset()
    local veh = g_currentMission and g_currentMission.controlledVehicle
    if veh and veh.spec_tensionBelts then
        local spec = veh.spec_tensionBelts
        if veh.freeTensionBeltObject then
            for objectId, entry in pairs(spec.objectsToJoint or {}) do
                pcall(veh.freeTensionBeltObject, veh, objectId, spec.objectsToJoint, spec.isDynamic, entry and entry.object or nil)
            end
        end
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Hard reset done") end
    else
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] No controlled vehicle with tensionBelts") end
    end
end
addConsoleCommand("gsRMCBeltAnchorHardReset", "Force-free all RMC belt anchors on the controlled vehicle", "_hardReset", RMCBeltAnchor)

function RMCBeltAnchor.prerequisitesPresent(specs)
    return SpecializationUtil.hasSpecialization(TensionBelts, specs)
end

function RMCBeltAnchor.registerEventListeners(vType)
    SpecializationUtil.registerOverwrittenFunction(vType, "lockTensionBeltObject", RMCBeltAnchor.lockTensionBeltObject_OW)
    SpecializationUtil.registerOverwrittenFunction(vType, "freeTensionBeltObject", RMCBeltAnchor.freeTensionBeltObject_OW)
    SpecializationUtil.registerOverwrittenFunction(vType, "getAllowDynamicMountObjects", RMCBeltAnchor.getAllowDynamicMountObjects_OW)
    SpecializationUtil.registerEventListener(vType, "onLoad", RMCBeltAnchor)
    SpecializationUtil.registerEventListener(vType, "onDelete", RMCBeltAnchor)
    SpecializationUtil.registerEventListener(vType, "onUpdate", RMCBeltAnchor)
end

function RMCBeltAnchor:onLoad()
    self.spec_rmcBeltAnchor = { entries = {} }
end

function RMCBeltAnchor:onDelete()
    local s = self.spec_rmcBeltAnchor
    if not s or not s.entries then return end
    for objectId, e in pairs(s.entries) do
        if e.rmcMultiJoint and e.rmcMultiJoint.joints then
            for _,j in ipairs(e.rmcMultiJoint.joints) do pcall(removeJoint, j) end
        end
        if e.rmcMultiJoint and e.rmcMultiJoint.anchors then
            for _,a in ipairs(e.rmcMultiJoint.anchors) do if a and entityExists(a) then pcall(delete,a) end end
        end
        if e.rmcKinematic and entityExists(objectId) then
            link(getRootNode(), objectId)
            setRigidBodyType(objectId, RigidBodyType.DYNAMIC)
        end
        if e.rmcMassOrig and entityExists(objectId) and self.isServer then
            pcall(setMass, objectId, e.rmcMassOrig)
        end
    end
    self.spec_rmcBeltAnchor.entries = {}
end

function RMCBeltAnchor:onUpdate(dt)
    -- Mirror base TensionBelts replay queue: track if beltsToLoad is active
    local tb = self.spec_tensionBelts
    local rmc = self.spec_rmcBeltAnchor
    if tb ~= nil then
        local active = (tb.beltsToLoad ~= nil and #tb.beltsToLoad > 0) or false
        if rmc == nil then
            self.spec_rmcBeltAnchor = { entries = {}, loadQueueActive = active }
        else
            rmc.loadQueueActive = active
        end
    end
end


function RMCBeltAnchor.getAllowDynamicMountObjects_OW(self, superFunc)
    local rmc = self.spec_rmcBeltAnchor
    if rmc and rmc.blockDM then
        return false
    end
    if superFunc ~= nil then
        return superFunc(self)
    end
    return true
end


local function _newRigidJoint(jointNode, objectId, offX, offY, offZ, k, d)
    local jc = JointConstructor.new()
    jc:setActors(jointNode, objectId)

    local anchor = createTransformGroup("rmcBeltAnchor")
    link(jointNode, anchor)
    local wx, wy, wz = localToWorld(objectId, offX, offY, offZ)
    setWorldTranslation(anchor, wx, wy, wz)

    jc:setJointTransforms(anchor, anchor)
    jc:setEnableCollision(true)

    jc:setRotationLimit(0, 0, 0)
    jc:setRotationLimit(1, 0, 0)
    jc:setRotationLimit(2, 0, 0)
    jc:setRotationLimitSpring(RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                              RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                              RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D)
    jc:setTranslationLimitSpring(RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                                 RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D,
                                 RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D)

    local j = jc:finalize()
    return j, anchor
end

function RMCBeltAnchor.lockTensionBeltObject_OW(self, superFunc, objectId, objectsToJointTable, isDynamic, jointNode, object)
    -- SUPPRESS base lock entirely. One object => one joint set.
    local rmc = self.spec_rmcBeltAnchor or {}
    rmc.entries = rmc.entries or {}
    self.spec_rmcBeltAnchor = rmc

    if objectsToJointTable[objectId] ~= nil then
        return
    end

    -- RMC: detect if this strap call occurs during base replay queue
    local tb = self.spec_tensionBelts
    local inQueue = (tb ~= nil and tb.beltsToLoad ~= nil and #tb.beltsToLoad > 0) or false
    local rmc = self.spec_rmcBeltAnchor or {}
    self.spec_rmcBeltAnchor = rmc
    rmc.inQueue = inQueue


    -- RMC: split objects ignore our script; delegate to base strap logic
    if getSplitType(objectId) ~= 0 then
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Lock: %s  (SPLIT) -> base", tostring(self.configFileName or self.typeName))) end
        return superFunc(self, objectId, objectsToJointTable, isDynamic, jointNode, object)
    end

    -- Non-split path: enable our stiff multi-joint behavior
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Lock: %s  mode=%s  jointNode=%s", tostring(self.configFileName or self.typeName), tostring(RMCBeltAnchor.MODE), getName(jointNode))) end
    local rmc = self.spec_rmcBeltAnchor or {}
    self.spec_rmcBeltAnchor = rmc
    rmc.blockDM = true
    if object ~= nil and object.unmountKinematic ~= nil then
        pcall(function() object:unmountKinematic() end)
    end
    if self.isServer and entityExists(objectId) then
        link(getRootNode(), objectId)
        setRigidBodyType(objectId, RigidBodyType.DYNAMIC)
    end


    if RMCBeltAnchor.MODE == "KINEMATIC_LOCK" then
        if object and object.mountKinematic and (object.getSupportsMountKinematic == nil or object:getSupportsMountKinematic()) then
            object:mountKinematic(self, jointNode, 0,0,0, 0,0,0)
        else
            setRigidBodyType(objectId, RigidBodyType.KINEMATIC)
            local jx,jy,jz = getWorldTranslation(jointNode)
            local rx,ry,rz = getWorldRotation(jointNode)
            link(jointNode, objectId)
            setWorldTranslation(objectId, jx,jy,jz)
            setWorldRotation(objectId, rx,ry,rz)
        end
        objectsToJointTable[objectId] = { parent=getParent(objectId), object=object, rmcKinematic=true }
        rmc.entries[objectId] = objectsToJointTable[objectId]
        return
    end

    local created, anchors = {}, {}
    if self.isServer then
        for _,o in ipairs(RMCBeltAnchor.OFFSETS) do
            local jIdx, anc = _newRigidJoint(jointNode, objectId, o[1],o[2],o[3], RMCBeltAnchor.SPRING_K, RMCBeltAnchor.SPRING_D)
            table.insert(created, jIdx)
            table.insert(anchors, anc)
        end
        if getSplitType(objectId) ~= 0 then
            setInertiaScale(objectId,20,20,20)
        end

        -- Mass scale for non-split objects
        if getSplitType(objectId) == 0 and math.abs((RMCBeltAnchor.MASS_SCALE or 1.0) - 1.0) > 0.001 then
            local origMass = getMass(objectId) or 0
            if origMass > 0 then
                objectsToJointTable[objectId] = objectsToJointTable[objectId] or {}
                objectsToJointTable[objectId].rmcMassOrig = origMass
                local newMass = math.max(0.001, origMass * RMCBeltAnchor.MASS_SCALE)
                pcall(setMass, objectId, newMass)
            end
        end
    end

    objectsToJointTable[objectId] = objectsToJointTable[objectId] or {}
    objectsToJointTable[objectId].jointIndex = created[1]
    objectsToJointTable[objectId].jointTransform = anchors[1]
    objectsToJointTable[objectId].rmcMultiJoint = { joints=created, anchors=anchors }
    objectsToJointTable[objectId].object = object

    rmc.entries[objectId] = objectsToJointTable[objectId]
end

function RMCBeltAnchor.freeTensionBeltObject_OW(self, superFunc, objectId, objectsToJointTable, isDynamic, object)
    local _isSplitFree = (entityExists(objectId) and (getSplitType(objectId) ~= 0)) or false
    local entry = objectsToJointTable[objectId]
    local rmc = self.spec_rmcBeltAnchor

    if entry and entry.rmcMultiJoint then
        if self.isServer then
            local extras = entry.rmcMultiJoint
            local primary = entry.jointIndex
            for _, j in ipairs(extras.joints or {}) do
                if j ~= primary then pcall(removeJoint, j) end
            end
            for _, a in ipairs(extras.anchors or {}) do
                if a ~= entry.jointTransform and a and entityExists(a) then pcall(delete, a) end
            end
            if _isSplitFree then setInertiaScale(objectId,1,1,1) end
            if entry.rmcMassOrig and entityExists(objectId) then pcall(setMass, objectId, entry.rmcMassOrig) end
        end
        local res = superFunc(self, objectId, objectsToJointTable, isDynamic, object) -- base removes primary & resets flags
        if rmc and rmc.entries then rmc.entries[objectId] = nil; rmc.blockDM = next(rmc.entries) ~= nil end
if RMCBeltAnchor.DEBUG then print(string.format("[RMC_BeltAnchor] Free: %s", tostring(self.configFileName or self.typeName))) end
        return res
    end

    if entry and entry.rmcKinematic then
        if entry.object and entry.object.unmountKinematic then
            pcall(entry.object.unmountKinematic, entry.object, self)
        else
            if entityExists(objectId) then
                link(getRootNode(), objectId)
                setRigidBodyType(objectId, RigidBodyType.DYNAMIC)
            end
        end
        local res = superFunc(self, objectId, objectsToJointTable, isDynamic, object)
        if rmc and rmc.entries then rmc.entries[objectId] = nil end
        return res
    end

    
    -- RMC cleanup: ensure objects are fully releasable after strap release
    do
        local rmc = self.spec_rmcBeltAnchor
        local e = rmc and rmc.entries and rmc.entries[objectId]
        if e ~= nil then
            if e.joints ~= nil then
                for i=1, #e.joints do
                    local jid = e.joints[i]
                    if jid ~= nil then removeJoint(jid) end
                end
                e.joints = nil
            end
            if e.massSet ~= nil and e.originalMass ~= nil and entityExists(objectId) then
                pcall(setMass, objectId, e.originalMass)
                e.massSet = nil
                e.originalMass = nil
            end
            if e.inertiaScaled and entityExists(objectId) then
                pcall(setInertiaScale, objectId, 1,1,1)
                e.inertiaScaled = false
            end
            if e.wasKinematic and entityExists(objectId) then
                pcall(setRigidBodyType, objectId, RigidBodyType.DYNAMIC)
                e.wasKinematic = false
            end
            if e.wasMounted and object ~= nil and object.setDynamicMountType ~= nil then
                pcall(object.setDynamicMountType, object, DynamicMountUtil.MOUNT_TYPE_NONE, nil, true)
                e.wasMounted = false
            end
            self.spec_rmcBeltAnchor.entries[objectId] = nil
        end
    end

    return superFunc(self, objectId, objectsToJointTable, isDynamic, object)
end
if RMCBeltAnchor.DEBUG then print("[RMC_BeltAnchor] Loaded v1.1.0.0 (mass scaling on strap; default 0.85). Use 'gsRMCBeltAnchorMassScale <value>'.") end