-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

local thMain = g_thDesignKit
THPlaceableChangeConfigEvent = {}
local THPlaceableChangeConfigEvent_mt = THUtils.createClass(THPlaceableChangeConfigEvent, Event)
THUtils.initClass(THPlaceableChangeConfigEvent, "THPlaceableChangeConfigEvent", THClassType.EVENT)
function THPlaceableChangeConfigEvent.emptyNew()
    return Event.new(THPlaceableChangeConfigEvent_mt)
end
function THPlaceableChangeConfigEvent.new(placeable, configs, configData, boughtConfigs, isPurchased, purchasePrice, farmId)
    local self = THPlaceableChangeConfigEvent.emptyNew()
    local configManager = g_placeableConfigurationManager
    purchasePrice = purchasePrice or 0
    self.isEventValid = false
    if THUtils.argIsValid(THUtils.getIsType(placeable, Placeable), "placeable", placeable)
        and THUtils.argIsValid(configs == nil or type(configs) == THValueType.TABLE, "configs", configs)
        and THUtils.argIsValid(configData == nil or type(configData) == THValueType.TABLE, "configData", configData)
        and THUtils.argIsValid(boughtConfigs == nil or type(boughtConfigs) == THValueType.TABLE, "boughtConfigs", "boughtConfigs")
        and THUtils.argIsValid(not isPurchased or isPurchased == true, "isPurchased", isPurchased)
        and THUtils.argIsValid(type(purchasePrice) == THValueType.NUMBER and purchasePrice >= 0, "purchasePrice", purchasePrice)
        and THUtils.argIsValid(farmId == nil or THUtils.getIsType(farmId, THValueType.INTEGER), "farmId", farmId)
    then
        if placeable.storeItem == nil or placeable.storeItem.xmlFilename == nil then
            THUtils.errorMsg(true, "No store item exists for placeable")
        else
            configs = configs or placeable.configurations
            if configs ~= nil then
                self.placeable = placeable
                self.configurations = configs
                self.configurationData = configData or placeable.configurationData or {}
                self.isPurchased = isPurchased == true
                self.purchasePrice = purchasePrice
                self.farmId = farmId or AccessHandler.EVERYONE
                if boughtConfigs == nil then
                    self.boughtConfigurations = {}
                    self.hasBoughtConfigs = false
                    for configName, configId in pairs(configs) do
                        ConfigurationUtil.addBoughtConfiguration(configManager, self, configName, configId)
                    end
                else
                    self.boughtConfigurations = boughtConfigs
                    self.hasBoughtConfigs = true
                end
                self.isEventValid = true
            end
        end
    end
    return self
end
function THPlaceableChangeConfigEvent.readStream(self, streamId, connection)
    if streamReadBool(streamId) then
        local configManager = g_placeableConfigurationManager
        self.placeable = NetworkUtil.readNodeObject(streamId)
        self.isPurchased = streamReadBool(streamId)
        if self.isPurchased then
            self.purchasePrice = streamReadFloat32(streamId)
        else
            self.purchasePrice = 0
        end
        self.hasBoughtConfigs = streamReadBool(streamId)
        self.farmId = streamReadUIntN(streamId, FarmManager.FARM_ID_SEND_NUM_BITS)
        local storeItemXMLFilename = ""
        if self.placeable ~= nil and self.placeable.storeItem ~= nil then
            storeItemXMLFilename = THUtils.getNoNil(self.placeable.storeItem.xmlFilename, "")
        end
        self.configurations, self.boughtConfigurations, self.configurationData = ConfigurationUtil.readConfigurationsFromStream(configManager, streamId, connection, storeItemXMLFilename)
        self:run(connection)
    end
end
function THPlaceableChangeConfigEvent.writeStream(self, streamId, connection)
    if streamWriteBool(streamId, self.isEventValid == true) then
        local configManager = g_placeableConfigurationManager
        NetworkUtil.writeNodeObject(streamId, self.placeable)
        if streamWriteBool(streamId, self.isPurchased) then
            streamWriteFloat32(streamId, self.purchasePrice)
        end
        streamWriteBool(streamId, self.hasBoughtConfigs)
        streamWriteUIntN(streamId, self.farmId, FarmManager.FARM_ID_SEND_NUM_BITS)
        local storeItemXMLFilename = ""
        if self.placeable.storeItem ~= nil then
            storeItemXMLFilename = THUtils.getNoNil(self.placeable.storeItem.xmlFilename, "")
        end
        ConfigurationUtil.writeConfigurationsToStream(configManager, streamId, connection, storeItemXMLFilename, self.configurations, self.boughtConfigurations, self.configurationData)
    end
end
function THPlaceableChangeConfigEvent.run(self, connection)
    THUtils.call(function()
        local noEventSend = false
        if connection:getIsServer() then
            noEventSend = true
        end
        local boughtConfigs = nil
        local farmId = nil
        if self.hasBoughtConfigs then
            boughtConfigs = self.boughtConfigurations
        end
        if self.farmId ~= AccessHandler.EVERYONE then
            farmId = self.farmId
        end
        thMain:updatePlaceableConfigurations(self.placeable, self.configurations, self.configurationData, boughtConfigs, self.isPurchased, self.purchasePrice, farmId, noEventSend)
    end)
end
function THPlaceableChangeConfigEvent.sendEvent(placeable, configs, configData, boughtConfigs, isPurchased, purchasePrice, farmId)
    local newEvent = THPlaceableChangeConfigEvent.new(placeable, configs, configData, boughtConfigs, isPurchased, purchasePrice, farmId)
    local success = false
    if newEvent ~= nil and newEvent.isEventValid then
        success = true
        if g_server ~= nil then
            g_server:broadcastEvent(newEvent, false, nil, placeable)
        elseif g_client ~= nil then
            g_client:getServerConnection():sendEvent(newEvent)
        else
            success = false
        end
    end
    return success
end