-- Copyright 2024-2025 by Todd Hundersmarck (ThundR) 
-- All Rights Reserved

--[[

Unauthorized use and/or distribution of this work entitles
myself, the author, to unlimited free and unrestricted use,
access, and distribution of any works related to the unauthorized
user and/or distributor.

--]]

local thModPath = g_currentModDirectory
source(thModPath .. "shared/scripts/THCore.lua")
if g_thGlobalEnv.ConfigurationUtil.SEND_NUM_BITS < 16 then
    g_thGlobalEnv.ConfigurationUtil.SEND_NUM_BITS = 16
end
THDesignKit = {}
local THDesignKit_mt = THUtils.createClass(THDesignKit, THCore)
THDesignKit.NAME = "thDesignKit"
THDesignKit.CLASS_NAME = "THDesignKit"
THDesignKit.DATA_KEY = tostring(THDesignKit)
THDesignKit.NUM_CONFIG_SLOTS = 49 -- plus 1 for the non numbered variant
THDesignKit.OBJECT_TYPE = {
    VEHICLE = 1,
    PLACEABLE = 2
}
THUtils.createEnumTable(THDesignKit.OBJECT_TYPE)
THDesignKit.CONFIG_TYPE = {
    DESIGN = 1,
    COLOR = 2
}
THUtils.createEnumTable(THDesignKit.CONFIG_TYPE)
THDesignKit.OBJECT_CHANGE_TYPE = {
    VISIBILITY = "visibility",
    TRANSLATION = "translation",
    ROTATION = "rotation",
    SCALE = "scale",
    CENTER_OF_MASS = "centerOfMass"
}
THUtils.createEnumTable(THDesignKit.OBJECT_CHANGE_TYPE)
local function initScript()
    local self = THDesignKit.new(THDesignKit.NAME)
    if self ~= nil then
        _G.g_thDesignKit = self
    end
end
function THDesignKit.new(name, customMt)
    customMt = customMt or THDesignKit_mt
    local self = THCore.new(name, customMt)
    if self ~= nil then
        self.configManagers = {
            g_vehicleConfigurationManager,
            g_placeableConfigurationManager
        }
        self.configTypes = {}
        for _, objectTypeIndex in pairs(THDesignKit.OBJECT_TYPE) do
            self.configTypes[objectTypeIndex] = {}
            for configTypeId, configTypeIndex in pairs(THDesignKit.CONFIG_TYPE:getValues()) do
                local configTypeData = {
                    id = configTypeId,
                    index = configTypeIndex,
                    manager = self.configManagers[objectTypeIndex],
                    objectTypeIndex = objectTypeIndex
                }
                if configTypeIndex == THDesignKit.CONFIG_TYPE.DESIGN then
                    configTypeData.name = "design"
                    configTypeData.configName = "thDesign"
                    if objectTypeIndex == THDesignKit.OBJECT_TYPE.VEHICLE then
                        configTypeData.typeClass = VehicleConfigurationItem
                    elseif objectTypeIndex == THDesignKit.OBJECT_TYPE.PLACEABLE then
                        configTypeData.typeClass = PlaceableConfigurationItem
                    end
                elseif configTypeIndex == THDesignKit.CONFIG_TYPE.COLOR then
                    configTypeData.name = "color"
                    configTypeData.configName = "thColor"
                    if objectTypeIndex == THDesignKit.OBJECT_TYPE.VEHICLE then
                        configTypeData.typeClass = VehicleConfigurationItemColor
                    elseif objectTypeIndex == THDesignKit.OBJECT_TYPE.PLACEABLE then
                        configTypeData.typeClass = PlaceableConfigurationItemColor
                    end
                end
                if configTypeData.name ~= nil and configTypeData.typeClass ~= nil then
                    configTypeData.title = g_i18n:getText("thConfigTitle_" .. configTypeData.name)
                    self.configTypes[objectTypeIndex][configTypeIndex] = configTypeData
                end
            end
        end
        self.objectChangeNodes = {}
        self.areConfigsInitialized = false
        return self
    end
end
function THDesignKit.onPreInit(self, ...)
    THUtils.callSuperClass(THDesignKit, "onPreInit", self, ...)
    local modPath = self.coreData.mod.path
    THUtils.setFunctionHook(g_thGlobalEnv, "setShaderParameter", false, false, self, THDesignKit.inj_setShaderParameter)
    THUtils.setFunctionHook("SpecializationManager", "initSpecializations", false, false, self, THDesignKit.inj_initSpecializations)
    THUtils.setFunctionHook("PlaceableConfigurationItemColor", "registerXMLPaths", false, false, self, THDesignKit.inj_registerPlaceableColorXMLPaths)
    THUtils.setFunctionHook("PlaceableConfigurationItemColor", "onPostLoad", false, false, self, THDesignKit.inj_onPostLoadPlaceableColor)
    THUtils.setFunctionHook("ObjectChangeUtil", "loadValuesFromXML", false, false, self, THDesignKit.inj_loadObjectChangeValuesFromXML)
    THUtils.setFunctionHook("ObjectChangeUtil", "setObjectChange", false, false, self, THDesignKit.inj_setObjectChange)
    source(modPath .. "scripts/events/THPlaceableChangeConfigEvent.lua")
    source(modPath .. "scripts/gui/THConstructionScreen.lua")
end
function THDesignKit.loadConfigurations(self)
    local objectTypesList = THDesignKit.OBJECT_TYPE:getValues()
    for _, objectTypeIndex in pairs(objectTypesList) do
        local configManager = self.configManagers[objectTypeIndex]
        if configManager ~= nil then
            if objectTypeIndex == THDesignKit.OBJECT_TYPE.PLACEABLE then
                for _, configTypeData in ipairs(self.configTypes[objectTypeIndex]) do
                    for slotNum = 1, THDesignKit.NUM_CONFIG_SLOTS do
                        local configName = string.format("%s%d", configTypeData.configName, slotNum)
                        if slotNum == 1 then
                            configManager:addConfigurationType(configTypeData.configName, configTypeData.title, self.coreData.xmlKey, configTypeData.typeClass)
                        end
                        configManager:addConfigurationType(configName, configTypeData.title, self.coreData.xmlKey, configTypeData.typeClass)
                    end
                end
                for slotNum = 1, 8 do
                    local configName = string.format("thPlaceableDesign%d", slotNum)
                    local configTitle = g_i18n:getText("thConfigTitle_design")
                    configManager:addConfigurationType(configName, configTitle, self.coreData.xmlKey, PlaceableConfigurationItem)
                    configName = string.format("thPlaceableColor%d", slotNum)
                    configTitle = g_i18n:getText("thConfigTitle_color")
                    configManager:addConfigurationType(configName, configTitle, self.coreData.xmlKey, PlaceableConfigurationItemColor)
                end
            elseif objectTypeIndex == THDesignKit.OBJECT_TYPE.VEHICLE then
                local vehicleConfigNames = {
                    "design",
                    "designColor"
                }
                for _, baseConfigName in ipairs(vehicleConfigNames) do
                    local configTitle = g_i18n:getText("configuration_" .. baseConfigName)
                    for slotNum = 1, THDesignKit.NUM_CONFIG_SLOTS do
                        local configName = string.format(baseConfigName .. "%d", slotNum)
                        if configManager.configurations[configName] == nil then
                            configManager:addConfigurationType(configName, configTitle, nil, VehicleConfigurationItem)
                        end
                    end
                end
            end
        end
    end
    return true
end
function THDesignKit.updatePlaceableConfigurations(self, placeable, configs, configData, boughtConfigs, isPurchased, purchasePrice, farmId, noEventSend)
    purchasePrice = purchasePrice or 0
    local configManager = g_placeableConfigurationManager
    local mission = g_currentMission
    local success = false
    if THUtils.argIsValid(THUtils.getIsType(placeable, Placeable), "placeable", placeable)
        and THUtils.argIsValid(configs == nil or type(configs) == THValueType.TABLE, "configs", configs)
        and THUtils.argIsValid(configData == nil or type(configData) == THValueType.TABLE, "configData", configData)
        and THUtils.argIsValid(boughtConfigs == nil or type(boughtConfigs) == THValueType.TABLE, "boughtConfigs", boughtConfigs)
        and THUtils.argIsValid(type(purchasePrice) == THValueType.NUMBER and purchasePrice >= 0, "purchasePrice", purchasePrice)
    then
        isPurchased = THUtils.validateArg(not isPurchased or isPurchased == true, "isPurchased", isPurchased, false)
        success = true
        if mission ~= nil
            and (self.coreData.isServer or noEventSend == true)
        then
            configs = configs or placeable.configurations
            configData = configData or placeable.configurationData
            if boughtConfigs == nil then
                placeable:setConfigurations(configs, placeable.boughtConfigurations or {}, configData)
            else
                placeable:setConfigurations(configs, boughtConfigs, configData)
            end
            if isPurchased then
                if self.coreData.isServer then
                    THUtils.clearTable(placeable.boughtConfigurations)
                    for configName, configId in pairs(placeable.configurations) do
                        ConfigurationUtil.addBoughtConfiguration(configManager, placeable, configName, configId)
                    end
                    boughtConfigs = placeable.boughtConfigurations
                    farmId = farmId or placeable:getOwnerFarmId()
                    if purchasePrice > 0 and farmId ~= nil
                        and farmId ~= AccessHandler.EVERYONE
                        and farmId ~= AccessHandler.NOBODY
                        and farmId ~= FarmManager.SPECTATOR_FARM_ID
                    then
                        mission:addMoney(-purchasePrice, farmId, MoneyType.PROPERTY_MAINTENANCE, true, true)
                    end
                end
            end
            ConfigurationUtil.raiseConfigurationItemEvent(placeable, "onLoad")
            ConfigurationUtil.raiseConfigurationItemEvent(placeable, "onPostLoad")
            ConfigurationUtil.raiseConfigurationItemEvent(placeable, "onLoadFinished")
        end
        if noEventSend ~= true then
            success = THPlaceableChangeConfigEvent.sendEvent(placeable, configs, configData, boughtConfigs, isPurchased, purchasePrice, farmId)
        end
    end
    return success
end
function THDesignKit.setObjectChangeDefaults(self, object, apply)
    if THUtils.argIsValid(type(object) == THValueType.TABLE, "object", object)
        and THUtils.argIsValid(not apply or apply == true, "apply", apply)
    then
        local nodeId = object.node
        if THUtils.getIsType(object.parent, Placeable)
            and THUtils.getNoNil(nodeId, 0) > 0
            and object.values ~= nil and next(object.values) ~= nil
        then
            local xmlFilename = object.parent.configFileName
            local nodePath = I3DUtil.getNodePath(nodeId)
            if nodePath ~= nil and xmlFilename ~= nil
                and self.objectChangeNodes[xmlFilename] ~= nil
                and self.objectChangeNodes[xmlFilename][nodePath] ~= nil
            then
                local nodeData = self.objectChangeNodes[xmlFilename][nodePath]
                for _, valueDesc in pairs(object.values) do
                    local valueName = valueDesc.name
                    if valueName ~= nil and valueName ~= ""
                        and (valueDesc.active == nil or valueDesc.inactive == nil)
                    then
                        local srcValues = THUtils.getNoNil(valueDesc.active, valueDesc.inactive)
                        local defaultData = nodeData.defaultValues[valueName]
                        if defaultData == nil then
                            defaultData = {
                                isActive = false
                            }
                            nodeData.defaultValues[valueName] = defaultData
                            local srcType = type(srcValues)
                            local tgtValues = nil
                            if srcType == THValueType.TABLE then
                                tgtValues = THUtils.copyTable(srcValues)
                            else
                                tgtValues = { srcValues }
                            end
                            if next(tgtValues) ~= nil then
                                if valueName == THDesignKit.OBJECT_CHANGE_TYPE.VISIBILITY then
                                    local isVisible = getVisibility(nodeId)
                                    if isVisible ~= nil then
                                        tgtValues[1] = isVisible
                                        defaultData.isActive = true
                                    end
                                elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.TRANSLATION
                                    or valueName == THDesignKit.OBJECT_CHANGE_TYPE.ROTATION
                                    or valueName == THDesignKit.OBJECT_CHANGE_TYPE.SCALE
                                    or valueName == THDesignKit.OBJECT_CHANGE_TYPE.CENTER_OF_MASS
                                then
                                    local x, y, z = nil, nil, nil
                                    if valueName == THDesignKit.OBJECT_CHANGE_TYPE.TRANSLATION then
                                        x, y, z = getTranslation(nodeId)
                                    elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.ROTATION then
                                        x, y, z = getRotation(nodeId)
                                    elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.SCALE then
                                        x, y, z = getScale(nodeId)
                                    elseif valueName == THDesignKit.OBJECT_CHANGE_TYPE.CENTER_OF_MASS then
                                        x, y, z = getCenterOfMass(nodeId)
                                    end
                                    if x ~= nil and y ~= nil and z ~= nil then
                                        tgtValues[1] = x
                                        tgtValues[2] = y
                                        tgtValues[3] = z
                                        defaultData.isActive = true
                                    end
                                end
                            end
                            if defaultData.isActive then
                                if srcType == THValueType.TABLE then
                                    defaultData.values = tgtValues
                                else
                                    defaultData.values = tgtValues[1]
                                end
                            end
                        end
                        if apply == true and defaultData.isActive then
                            if valueDesc.active == nil then
                                valueDesc.active = defaultData.values
                            elseif valueDesc.inactive == nil then
                                valueDesc.inactive = defaultData.values
                            end
                        end
                    end
                end
            end
        end
    end
end
function THDesignKit.inj_setShaderParameter(self, superFunc, nodeId, paramName, r, g, b, a, ...)
    if self.extendedConfigNodes ~= nil then
        THUtils.call(function()
            if THUtils.getNoNil(nodeId, 0) > 0 then
                local nodePath = I3DUtil.getNodePath(nodeId)
                local nodeData = self.extendedConfigNodes[nodePath]
                if nodeData ~= nil then
                    if nodeData.shaderMapping ~= nil then
                        local newParamName = nodeData.shaderMapping[paramName]
                        if newParamName ~= nil then
                            paramName = newParamName
                        end
                    end
                end
            end
        end)
    end
    return superFunc(nodeId, paramName, r, g, b, a, ...)
end
function THDesignKit.inj_initSpecializations(self, superFunc, manager, ...)
    if not self.areConfigsInitialized then
        THUtils.pcall(function()
            self:loadConfigurations()
        end)
        self.areConfigsInitialized = true
    end
    return superFunc(manager, ...)
end
function THDesignKit.inj_registerPlaceableColorXMLPaths(self, superFunc, xmlSchema, xmlRootPath, xmlPath, ...)
    local function appendFunc(...)
        if xmlSchema ~= nil and xmlRootPath ~= nil then
            THUtils.call(function()
                local xmlSubPath = xmlRootPath .. "." .. self.coreData.xmlKey
                local nodeParamPath = xmlSubPath .. ".nodeParams.node(?)"
                xmlSchema:register(XMLValueType.NODE_INDEX, nodeParamPath .. "#node", "Node id")
                xmlSchema:register(XMLValueType.STRING, nodeParamPath .. "#shader", "Shader parameter")
            end)
        end
        return ...
    end
    return appendFunc(superFunc(xmlSchema, xmlRootPath, xmlPath, ...))
end
function THDesignKit.inj_onPostLoadPlaceableColor(self, superFunc, item, placeable, ...)
    if placeable ~= nil then
        THUtils.call(function()
            local wrappedXML = THUtils.wrapXMLFile(placeable.xmlFile)
            local components = placeable.components
            local i3dMappings = placeable.i3dMappings
            local objectTypeIndex = THDesignKit.OBJECT_TYPE.PLACEABLE
            if wrappedXML ~= nil then
                local configManager = self.configManagers[objectTypeIndex]
                local configDesc = configManager:getConfigurationDescByName(item.configName)
                if configDesc ~= nil then
                    local itemData = THUtils.createDataTable(item, THDesignKit.DATA_KEY)
                    if itemData ~= nil then
                        if itemData.configNodes == nil then
                            itemData.configNodes = {}
                        else
                            THUtils.clearTable(itemData.configNodes)
                        end
                        local xmlSubKey = configDesc.configurationKey .. "." .. self.coreData.xmlKey
                        wrappedXML:iterate(xmlSubKey .. ".nodeParams.node", function(_, pNodeKey)
                            local nodeId = wrappedXML:getValue(pNodeKey .. "#node", nil, components, i3dMappings)
                            if THUtils.getNoNil(nodeId, 0) <= 0 then
                                THUtils.xmlErrorMsg(pNodeKey, nil, THMessage.INVALID_VALUE, "node", nodeId)
                            else
                                local nodePath = I3DUtil.getNodePath(nodeId)
                                local shaderParam = wrappedXML:getValue(pNodeKey .. "#shader")
                                if nodePath ~= nil then
                                    local nodeData = itemData.configNodes[nodePath]
                                    if nodeData == nil then
                                        nodeData = {
                                            shaderMapping = {}
                                        }
                                        itemData.configNodes[nodePath] = nodeData
                                    end
                                    if shaderParam ~= nil and shaderParam ~= "" then
                                        if not getHasShaderParameter(nodeId, shaderParam) then
                                            THUtils.xmlErrorMsg(pNodeKey, nil, THMessage.INVALID_VALUE, "shader", shaderParam)
                                        else
                                            nodeData.shaderMapping["colorScale0"] = shaderParam
                                        end
                                    end
                                end
                            end
                        end)
                        self.extendedConfigNodes = itemData.configNodes
                    end
                end
            end
        end)
    end
    local function appendFunc(...)
        self.extendedConfigNodes = nil
        return ...
    end
    return appendFunc(superFunc(item, placeable, ...))
end
function THDesignKit.inj_loadObjectChangeValuesFromXML(self, superFunc, xmlFile, xmlKey, nodeId, object, target, ...)
    THUtils.call(function()
        if THUtils.getIsType(target, Placeable)
            and THUtils.getNoNil(nodeId, 0) > 0
            and xmlKey ~= nil and string.find(xmlKey, "Configurations.", nil, true)
        then
            local xmlFilename = target.configFileName
            local nodePath = I3DUtil.getNodePath(nodeId)
            if nodePath ~= nil
                and xmlFilename ~= nil and xmlFilename ~= ""
            then
                local targetNodes = self.objectChangeNodes[xmlFilename]
                if targetNodes == nil then
                    targetNodes = {}
                    self.objectChangeNodes[xmlFilename] = targetNodes
                end
                local nodeData = targetNodes[nodePath]
                if nodeData == nil then
                    nodeData = {
                        defaultValues = {}
                    }
                    targetNodes[nodePath] = nodeData
                end
            end
        end
    end)
    return superFunc(xmlFile, xmlKey, nodeId, object, target, ...)
end
function THDesignKit.inj_setObjectChange(self, superFunc, object, isActive, target, ...)
    if object ~= nil and target ~= nil then
        THUtils.call(function()
            self:setObjectChangeDefaults(object, true)
        end)
    end
    return superFunc(object, isActive, target, ...)
end
THUtils.pcall(initScript)