MS2Telemetry = {
	LOG_ENABLED = false
}

local FSContext = {
	UpdateInterval = {
		Target = 16.66,
		Current = 0.0
	},
	PipeControl = {
		Pipe = nil,
		PipeName = "\\\\.\\pipe\\ms2fstelemetry",
		RefreshRate = 300,
		RefreshCurrent = -1
	},
	MaxDepthImplements = 10,
	Telemetry = {}
};

function MS2Telemetry:loadMap(name)
	self:Log("Entering loadMap with name: " .. tostring(name));

	MS2Telemetry:ClearGameTelemetry();
	MS2Telemetry:ClearVehicleTelemetry();
	MS2Telemetry:ProcessGameEdition();
end;

function MS2Telemetry:update(dt)
	FSContext.UpdateInterval.Current = FSContext.UpdateInterval.Current + dt;
	if FSContext.UpdateInterval.Current >= FSContext.UpdateInterval.Target then
		FSContext.UpdateInterval.Current = 0;

		MS2Telemetry:RefreshPipe();

		if FSContext.PipeControl.Pipe ~= nil then
			FSContext.Telemetry.IsDrivingVehicle = MS2Telemetry:IsDrivingVehicle();

            MS2Telemetry:ClearHudData();
			
            if FSContext.Telemetry.IsDrivingVehicle then
				MS2Telemetry:ProcessVehicleData();
			else
				MS2Telemetry:ClearVehicleTelemetry();
			end;

			MS2Telemetry:ProcessGameData();
			MS2Telemetry:WriteTelemetry();
		else
			self:Log("Pipe is NIL.");
		end;
	end;
end

function MS2Telemetry:ClearGameTelemetry()
	FSContext.Telemetry.Money = 0.0;
	FSContext.Telemetry.TemperatureMin = 0.0;
	FSContext.Telemetry.TemperatureMax = 0.0;
	FSContext.Telemetry.TemperatureTrend = 0;
	FSContext.Telemetry.DayTimeMinutes = 0;
	FSContext.Telemetry.WeatherCurrent = 0;
	FSContext.Telemetry.WeatherNext = 0;
	FSContext.Telemetry.Day = 0;
	FSContext.Telemetry.GameEdition = 0;
	FSContext.Telemetry.SpeedUnit = 0;
end

function MS2Telemetry:ClearHudData()
    FSContext.Telemetry.AnimalIsRiding = false;
    FSContext.Telemetry.AnimalHudName = "";
    FSContext.Telemetry.AnimalHudDirt = 0.0;
    FSContext.Telemetry.AnimalHudFitness = 0.0;
    FSContext.Telemetry.AnimalHudHealth = 0.0;
    FSContext.Telemetry.AnimalHudRiding = 0.0;
    FSContext.Telemetry.AnimalHudAge = 0;
end

function MS2Telemetry:ClearVehicleTelemetry()
	FSContext.Telemetry.VehicleName = "";
	FSContext.Telemetry.FuelType = 0;  --0 Undefined | 1 Diesel | 2 Eletric | 3 Methane
	FSContext.Telemetry.FuelMax = 0.0;
	FSContext.Telemetry.Fuel = 0.0;
	FSContext.Telemetry.RPMMin = 0;
	FSContext.Telemetry.RPMMax = 0;
	FSContext.Telemetry.RPM = 0;
	FSContext.Telemetry.IsDrivingVehicle = false;
	FSContext.Telemetry.IsAiActive = false;
	FSContext.Telemetry.Wear = 0.0;
	FSContext.Telemetry.OperationTimeMinutes = 0;
	FSContext.Telemetry.Speed = 0.0;
	FSContext.Telemetry.IsEngineStarted = false;
	FSContext.Telemetry.Gear = 0;
    FSContext.Telemetry.GearName = "";
    FSContext.Telemetry.GearAvailable = false;
    FSContext.Telemetry.GearAutomatic = false;
    FSContext.Telemetry.GearChanging = false;
	FSContext.Telemetry.IsLightOn = false;
	FSContext.Telemetry.IsLightHighOn = false;
	FSContext.Telemetry.IsLightTurnRightEnabled = false;
	FSContext.Telemetry.IsLightTurnRightOn = false;
	FSContext.Telemetry.IsLightTurnLeftEnabled = false;
	FSContext.Telemetry.IsLightTurnLeftOn = false;
	FSContext.Telemetry.IsLightHazardOn = false;
	FSContext.Telemetry.IsLightBeaconOn = false;
	FSContext.Telemetry.IsWipersOn = false;
	FSContext.Telemetry.IsCruiseControlOn = false;
	FSContext.Telemetry.CruiseControlMaxSpeed = 0;
	FSContext.Telemetry.CruiseControlSpeed = 0;
	FSContext.Telemetry.IsHandBrakeOn = false;
	FSContext.Telemetry.IsReverseDriving = false;
	FSContext.Telemetry.IsMotorFanEnabled = false;
	FSContext.Telemetry.MotorTemperature = 0.0;
	FSContext.Telemetry.VehiclePrice = 0.0;
	FSContext.Telemetry.VehicleSellPrice = 0.0;
	FSContext.Telemetry.IsHonkOn = false;
	FSContext.Telemetry.AngleRotation = 0.0;
	FSContext.Telemetry.Mass = 0.0;
	FSContext.Telemetry.TotalMass = 0.0;
	FSContext.Telemetry.IsOnField = false;
	FSContext.Telemetry.DefMax = 0.0;
	FSContext.Telemetry.Def = 0.0;
	FSContext.Telemetry.AirMax = 0.0;
	FSContext.Telemetry.Air = 0.0;
    MS2Telemetry:ClearAttachedImplements();
    MS2Telemetry:ClearLoads();
end

function MS2Telemetry:ClearLoads()
    FSContext.Telemetry.LoadCapacity = {};
    FSContext.Telemetry.LoadQuantity  = {};
    FSContext.Telemetry.LoadType = {};
    FSContext.Telemetry.LoadContainerType  = {};
    FSContext.Telemetry.LoadContainerName  = {};
end

function MS2Telemetry:ClearAttachedImplements()
	FSContext.Telemetry.AttachedImplementsPosition = {};
	FSContext.Telemetry.AttachedImplementsLowered = {};
	FSContext.Telemetry.AttachedImplementsSelected = {};
	FSContext.Telemetry.AttachedImplementsTurnedOn = {};
	FSContext.Telemetry.AttachedImplementsWear = {};
end

function MS2Telemetry:IsDrivingVehicle()
	local vehicle = MS2Telemetry:GetVehicle();
	local hasVehicle = vehicle ~= nil;
	return hasVehicle and vehicle.spec_motorized ~= nil;
end

function MS2Telemetry:ProcessVehicleData()
	local mission = g_currentMission;
	local vehicle = MS2Telemetry:GetVehicle();
	local motor = vehicle:getMotor();
	local specMotorized = vehicle.spec_motorized;
	local specDrivable = vehicle.spec_drivable;
	local specLights = vehicle.spec_lights;
	local specWipers = vehicle.spec_wipers;
	local specHonk = vehicle.spec_honk;
	local specWearable = vehicle.spec_wearable;

	MS2Telemetry:ProcessPrice(vehicle);
	MS2Telemetry:ProcessMotorFanEnabled(specMotorized);
	MS2Telemetry:ProcessMotorTemperature(specMotorized);
	MS2Telemetry:ProcessSpeed(vehicle, specMotorized);
	MS2Telemetry:ProcessGear(motor);
	MS2Telemetry:ProcessRPM(motor);
	MS2Telemetry:ProcessReverseDriving(vehicle, specMotorized);
	MS2Telemetry:ProcessEngineStarted(specMotorized);
	MS2Telemetry:ProcessVehicleName(mission);
	MS2Telemetry:ProcessAiActive(vehicle);
	MS2Telemetry:ProcessWear(specWearable);
	MS2Telemetry:ProcessOperationTime(vehicle);
	MS2Telemetry:ProcessFuel(vehicle, specMotorized);
	MS2Telemetry:ProcessCruiseControl(specDrivable);
	MS2Telemetry:ProcessHandBrake(specDrivable);
	MS2Telemetry:ProcessTurnLightsHazard(specLights);
	MS2Telemetry:ProcessLightBeacon(specLights);
	MS2Telemetry:ProcessLight(specLights);
	MS2Telemetry:ProcessWiper(specWipers, mission);
	MS2Telemetry:ProcessHonk(specHonk);

	MS2Telemetry:ClearAttachedImplements();
	MS2Telemetry:ProcessAttachedImplements(vehicle, false, 0, 0);

	MS2Telemetry:ProcessAngleRotation(vehicle);
	MS2Telemetry:ProcessMass(vehicle);
	MS2Telemetry:ProcessOnField(vehicle);
	MS2Telemetry:ProcessDef(vehicle);
	MS2Telemetry:ProcessAir(vehicle);

    MS2Telemetry:ClearLoads();
    MS2Telemetry:ProcessLoads(vehicle);
end

function MS2Telemetry:ProcessLoads(vehicle)
    local loads = MS2Telemetry:GetAllLoads(vehicle);

    for index, load in ipairs(loads) do
        FSContext.Telemetry.LoadCapacity[index] = load.capacity
        FSContext.Telemetry.LoadQuantity[index] = load.quantity
        FSContext.Telemetry.LoadType[index] = load.type
        FSContext.Telemetry.LoadContainerType[index] = load.typeDesc
        FSContext.Telemetry.LoadContainerName[index] = load.typeName
    end
end

function MS2Telemetry:ProcessHudData(player)
    local isRiding = false;
    local hud = player.hudUpdater;
    local vehicle = MS2Telemetry:GetVehicle();

    if vehicle ~= nil then
        local specRideable = vehicle.spec_rideable;
        isRiding = specRideable ~= nil;
    end

    if hud ~= nil then
        if hud.isAnimal then
            if hud.object ~= nil then
                FSContext.Telemetry.AnimalIsRiding = isRiding;
                FSContext.Telemetry.AnimalHudName = hud.object.name;
                FSContext.Telemetry.AnimalHudDirt = hud.object.dirt;
                FSContext.Telemetry.AnimalHudFitness = hud.object.fitness;
                FSContext.Telemetry.AnimalHudHealth = hud.object.health;
                FSContext.Telemetry.AnimalHudRiding = hud.object.riding;
                FSContext.Telemetry.AnimalHudAge = hud.object.age;
            end
        end
    end
end

function MS2Telemetry:ProcessAttachedImplements(vehicle, invertX, x, depth)
-- Indexes:
-- Negative: Front of vehicle
-- 0: tractor front loader
-- Positive: Back of vehicle
-- Smaller absolute value: Closer to vehicle

	local attachedImplements = vehicle:getAttachedImplements();
	if attachedImplements == nil then
		return;
	end

    for _, implement in pairs(attachedImplements) do
		local object = implement.object
		if object ~= nil and object.schemaOverlay ~= nil then
			local wear = object.getDamageAmount ~= nil and object:getDamageAmount() or 0.0;
			local selected = object:getIsSelected()
            local turnedOn = object.getIsTurnedOn ~= nil and object:getIsTurnedOn()
			local lowered = object.getIsLowered ~= nil and object:getIsLowered(true);
            local jointDesc = vehicle.schemaOverlay.attacherJoints[implement.jointDescIndex];
			if jointDesc ~= nil then
				local isInverted  = invertX ~= jointDesc.invertX
                local baseX
                if isInverted  then
                    baseX = x - 1 + (1 - jointDesc.x)
                else
                    baseX = x + jointDesc.x
                end
				baseX = math.ceil(baseX);

				FSContext.Telemetry.AttachedImplementsPosition[baseX] = baseX;
				FSContext.Telemetry.AttachedImplementsLowered[baseX] = lowered;
				FSContext.Telemetry.AttachedImplementsSelected[baseX] = selected;
				FSContext.Telemetry.AttachedImplementsTurnedOn[baseX] = turnedOn;
				FSContext.Telemetry.AttachedImplementsWear[baseX] = wear;
				if FSContext.MaxDepthImplements > depth then
					MS2Telemetry:ProcessAttachedImplements(object, invertX, baseX, depth + 1)
				end
			end
		end
	end
end

function MS2Telemetry:GetAllLoads(vehicle, processedIds)
    local allLoads = {}
    processedIds = processedIds or {}

	if g_fillTypeManager == nil then
		self:Log("Error: g_fillTypeManager is nil in GetAllLoads.")
		return allLoads;
	end	

    -- Process Fill Units
    local function processFillUnits(fillUnits, typeDesc, typeName)
        for fillUnitIndex, fillUnit in pairs(fillUnits) do
            local fillType = g_fillTypeManager:getFillTypeNameByIndex(fillUnit.fillType)
            table.insert(allLoads, {
                type = fillType,
                quantity = fillUnit.fillLevel,
                capacity = fillUnit.capacity,
                typeDesc = typeDesc,
                typeName = typeName,
            })
        end
    end

    -- Process Tension Belt Loads
    local function processTensionBeltsLoads(tensionBelts, typeDesc, typeName)
        if not tensionBelts.hasTensionBelts then return end
		for _, belt in pairs(tensionBelts.objectsToJoint) do
			if belt.object ~= nil and belt.object.spec_fillUnit ~= nil then
				processFillUnits(belt.object.spec_fillUnit.fillUnits, typeDesc, typeName)
			end
		end
    end

    -- Process Dynamically Mounted Objects
    local function processDynamicMountAttacherLoads(dynamicMountAttacher, typeDesc, typeName)
        for _, mountedObject in pairs(dynamicMountAttacher.dynamicMountedObjects) do
            if mountedObject.fillLevel and mountedObject.fillType then
                local fillType = g_fillTypeManager:getFillTypeNameByIndex(mountedObject.fillType)
                table.insert(allLoads, {
                    type = fillType,
                    quantity = mountedObject.fillLevel,
                    capacity = mountedObject.fillLevel, -- Temp, until I find out where this is
                    typeDesc = typeDesc,
                    typeName = typeName,
                })
            elseif mountedObject.spec_fillUnit then
                processFillUnits(mountedObject.spec_fillUnit.fillUnits, typeDesc, typeName)
            end
        end
    end


    -- Get typeDesc and typeName from vehicle if available
    local vehicleTypeDesc = vehicle.typeDesc or ""
    local vehicleTypeName = vehicle.typeName or ""
    local vehicleId = vehicle.networkId or vehicle.id -- Unique identifier for the vehicle, use networkId or any unique property

    if not processedIds[vehicleId] then
        processedIds[vehicleId] = true

        -- Process all specializations for the vehicle:
        if vehicle.spec_fillUnit then
            processFillUnits(vehicle.spec_fillUnit.fillUnits, vehicleTypeDesc, vehicleTypeName)
        end
        if vehicle.spec_tensionBelts then
            processTensionBeltsLoads(vehicle.spec_tensionBelts, vehicleTypeDesc, vehicleTypeName)
        end
        if vehicle.spec_dynamicMountAttacher then
            processDynamicMountAttacherLoads(vehicle.spec_dynamicMountAttacher, vehicleTypeDesc, vehicleTypeName)
        end
    end

    -- Check for attached implements (e.g., trailers)
    if vehicle.getAttachedImplements then
        for _, implement in pairs(vehicle:getAttachedImplements()) do
            local implementId = implement.object.networkId or implement.object.id
            local implementTypeDesc = implement.object.typeDesc or ""
            local implementTypeName = implement.object.typeName or ""

            if not processedIds[implementId] then
                processedIds[implementId] = true

                -- Process all specializations for the implement:
                if implement.object.spec_fillUnit then
                    processFillUnits(implement.object.spec_fillUnit.fillUnits, implementTypeDesc, implementTypeName)
                end
                if implement.object.spec_tensionBelts then
                    processTensionBeltsLoads(implement.object.spec_tensionBelts, vehicleTypeDesc, vehicleTypeName)
                end
                if implement.object.spec_dynamicMountAttacher then
                    processDynamicMountAttacherLoads(implement.object.spec_dynamicMountAttacher, vehicleTypeDesc, vehicleTypeName)
                end
            end

            -- Recursively check for implements attached to implements
            if implement.object.getAttachedImplements then
                local subImplementsLoads = MS2Telemetry:GetAllLoads(implement.object, processedIds)
                for _, load in pairs(subImplementsLoads) do
                    table.insert(allLoads, load)
                end
            end
        end
    end

    return allLoads
end

function MS2Telemetry:ProcessMotorFanEnabled(motorized)
	if motorized ~= nil and motorized.motorFan ~= nil then
		FSContext.Telemetry.IsMotorFanEnabled = motorized.motorFan.enabled;
	else
		FSContext.Telemetry.IsMotorFanEnabled = false;
	end;
end

function MS2Telemetry:ProcessMotorTemperature(motorized)
	if motorized ~= nil and motorized.motorTemperature ~= nil then
		FSContext.Telemetry.MotorTemperature = motorized.motorTemperature.value;
	else
		FSContext.Telemetry.MotorTemperature = 0;
	end;
end

function MS2Telemetry:ProcessSpeed(vehicle, motorized)
	if motorized ~= nil and vehicle.getLastSpeed ~= nil then
		FSContext.Telemetry.Speed = math.max(0.0, vehicle:getLastSpeed() * motorized.speedDisplayScale)
	else
		FSContext.Telemetry.Speed = 0;
	end;
end

function MS2Telemetry:ProcessRPM(motor)
	if motor ~= nil then
		if motor.getMinRpm ~= nil then
			FSContext.Telemetry.RPMMin = math.ceil(motor:getMinRpm());
		else
			FSContext.Telemetry.RPMMin = 0;
		end	

		if motor.getMaxRpm ~= nil then
			FSContext.Telemetry.RPMMax = math.ceil(motor:getMaxRpm());
		else
			FSContext.Telemetry.RPMMax = 0;
		end

		if motor.getLastRealMotorRpm ~= nil then
			FSContext.Telemetry.RPM = math.ceil(motor:getLastRealMotorRpm());
		else
			FSContext.Telemetry.RPM = 0;
		end
	end;
end

function MS2Telemetry:ProcessPrice(vehicle)
	if vehicle ~= nil then
		if vehicle.getPrice ~= nil then
			FSContext.Telemetry.VehiclePrice = vehicle:getPrice();
		else
			FSContext.Telemetry.VehiclePrice = 0.0;
		end

		if vehicle.getSellPrice ~= nil then
			FSContext.Telemetry.VehicleSellPrice = vehicle:getSellPrice();
		else
			FSContext.Telemetry.VehicleSellPrice = 0.0;
		end
	end;
end

function MS2Telemetry:ProcessGear(motor)
	if motor ~= nil then
   		FSContext.Telemetry.Gear = motor.gear;
        local gearName, available, isAutomatic, _prevGearName, _nextGearName, _prevPrevGearName, _nextNextGearName, isGearChanging = motor:getGearToDisplay();
        FSContext.Telemetry.GearName = gearName;
        FSContext.Telemetry.GearAvailable = available;
        FSContext.Telemetry.GearAutomatic = isAutomatic;
        FSContext.Telemetry.GearChanging = isGearChanging;
	end;
end

function MS2Telemetry:ProcessReverseDriving(vehicle, motorized)
	local reverserDirection = vehicle.getReverserDirection == nil and 1 or vehicle:getReverserDirection();
	FSContext.Telemetry.IsReverseDriving = vehicle:getLastSpeed() > motorized.reverseDriveThreshold and vehicle.movingDirection ~= reverserDirection;
end

function MS2Telemetry:ProcessEngineStarted(motorized)
	FSContext.Telemetry.IsEngineStarted = motorized ~= nil and motorized.isMotorStarted;
end

function MS2Telemetry:ProcessVehicleName(mission)
	FSContext.Telemetry.VehicleName = mission.currentVehicleName;
end

function MS2Telemetry:ProcessAiActive(vehicle)
	FSContext.Telemetry.IsAiActive = vehicle.getIsAIActive ~= nil and vehicle:getIsAIActive();
end

function MS2Telemetry:ProcessWear(wearable)
	if wearable ~= nil and wearable.damage ~= nil then
		FSContext.Telemetry.Wear = wearable.damage;
	else
		FSContext.Telemetry.Wear = 0.0;
	end;
end

function MS2Telemetry:ProcessOperationTime(vehicle)
	if vehicle.operatingTime ~= nil then
		FSContext.Telemetry.OperationTimeMinutes = math.floor(vehicle.operatingTime / (1000 * 60));
	else
		FSContext.Telemetry.OperationTimeMinutes = 0;
	end;
end

function MS2Telemetry:ProcessFuel(vehicle, motorized)
	FSContext.Telemetry.FuelType = 0;
	FSContext.Telemetry.FuelMax = 0;
	FSContext.Telemetry.Fuel = 0;

	if motorized == nil then
		return;
	end

	for _, consumer in pairs(motorized.consumersByFillTypeName) do		
		if consumer.fillType == FillType.DIESEL then
			FSContext.Telemetry.FuelType = 1;
		elseif consumer.fillType == FillType.ELECTRICCHARGE then
			FSContext.Telemetry.FuelType = 2;
		elseif consumer.fillType == FillType.METHANE then
			FSContext.Telemetry.FuelType = 3;
		end

		if FSContext.Telemetry.FuelType > 0 then
			if vehicle.getFillUnitCapacity ~= nil then
				FSContext.Telemetry.FuelMax = vehicle:getFillUnitCapacity(consumer.fillUnitIndex);
			end;
			if vehicle.getFillUnitFillLevel ~= nil then
				FSContext.Telemetry.Fuel = vehicle:getFillUnitFillLevel(consumer.fillUnitIndex);
			end;
			return;
		end
	end
end

function MS2Telemetry:GetPlayer()
	return g_currentMission.playerSystem:getLocalPlayer();
end

function MS2Telemetry:GetVehicle()
	return g_currentMission.playerSystem:getLocalPlayer():getCurrentVehicle();
end

function MS2Telemetry:ProcessCruiseControl(drivable)
	if drivable ~= nil then
		--Drivable.CRUISECONTROL_STATE_OFF
		--Drivable.CRUISECONTROL_STATE_ACTIVE
		--Drivable.CRUISECONTROL_STATE_FULL
		FSContext.Telemetry.IsCruiseControlOn = drivable.cruiseControl.state ~= Drivable.CRUISECONTROL_STATE_OFF;
		FSContext.Telemetry.CruiseControlSpeed = drivable.cruiseControl.speed;
		FSContext.Telemetry.CruiseControlMaxSpeed = drivable.cruiseControl.maxSpeed;
	else
		FSContext.Telemetry.IsCruiseControlOn = false;
		FSContext.Telemetry.CruiseControlSpeed = 0;
		FSContext.Telemetry.CruiseControlMaxSpeed = 0;
	end
end

--Aparently, it does'nt work
function MS2Telemetry:ProcessHandBrake(drivable)
	if drivable ~= nil then
		FSContext.Telemetry.IsHandBrakeOn = drivable.doHandbrake;
	else
		FSContext.Telemetry.IsHandBrakeOn = false;
	end
end

function MS2Telemetry:Clamp(value, min, max)
    return math.max(min, math.min(value, max));
end

function MS2Telemetry:ProcessTurnLightsHazard(lights)
	if lights ~= nil and lights.turnLightState ~= nil then
		local state = lights.turnLightState;
		FSContext.Telemetry.IsLightTurnRightEnabled = state == Lights.TURNLIGHT_RIGHT;
		FSContext.Telemetry.IsLightTurnLeftEnabled = state == Lights.TURNLIGHT_LEFT;
		FSContext.Telemetry.IsLightHazardOn = state == Lights.TURNLIGHT_HAZARD;
		local alpha = MS2Telemetry:Clamp((math.cos(7 * getShaderTimeSec()) + 0.2), 0, 1);
		FSContext.Telemetry.IsLightTurnRightOn = (FSContext.Telemetry.IsLightTurnRightEnabled or FSContext.Telemetry.IsLightHazardOn) and alpha > 0.5;
		FSContext.Telemetry.IsLightTurnLeftOn = (FSContext.Telemetry.IsLightTurnLeftEnabled or FSContext.Telemetry.IsLightHazardOn) and alpha > 0.5;
	else
		FSContext.Telemetry.IsLightTurnRightEnabled = false;
		FSContext.Telemetry.IsLightTurnRightOn = false;
		FSContext.Telemetry.IsLightTurnLeftEnabled = false;
		FSContext.Telemetry.IsLightTurnLeftOn = false;
		FSContext.Telemetry.IsLightHazardOn = false;
	end;
end

function MS2Telemetry:ProcessLightBeacon(lights)
	if lights ~= nil and lights.getBeaconLightsVisibility ~= nil then
		FSContext.Telemetry.IsLightBeaconOn = lights:getBeaconLightsVisibility();
	else
		FSContext.Telemetry.IsLightBeaconOn = false;
	end;
end

function MS2Telemetry:ProcessLight(lights)
	if lights ~= nil and lights.lightsTypesMask ~= nil then
		--0 - LIGHT				
		--1 - TURN LIGHT
		--2 - FRONTAL LIGHT
		--3 - HIGH LIGHT

		FSContext.Telemetry.IsLightOn = bitAND(lights.lightsTypesMask, 2^0) ~= 0;
		FSContext.Telemetry.IsLightHighOn = bitAND(lights.lightsTypesMask, 2^3) ~= 0;
	else
		FSContext.Telemetry.IsLightOn = false;
		FSContext.Telemetry.IsLightHighOn = false;
	end;
end

function MS2Telemetry:ProcessWiper(wipers, mission)
	FSContext.Telemetry.IsWipersOn = false;
	if wipers ~= nil and wipers.hasWipers then
		local rainScale = (mission.environment ~= nil and mission.environment.weather ~= nil and mission.environment.weather.getRainFallScale ~= nil) and mission.environment.weather:getRainFallScale() or 0;
		if rainScale > 0 then
			for _, wiper in pairs(wipers.wipers) do
				for stateIndex,state in ipairs(wiper.states) do
					if rainScale <= state.maxRainValue then
						FSContext.Telemetry.IsWipersOn = true;
						return
					end
				end
			end
		end
	end;
end

function MS2Telemetry:ProcessHonk(honk)
	FSContext.Telemetry.IsHonkOn = false;
	if honk ~= nil and honk.isPlaying ~= nil then
		FSContext.Telemetry.IsHonkOn = honk.isPlaying;
	end;
end

function MS2Telemetry:ProcessAngleRotation(vehicle)
	local x,_y,z = localDirectionToWorld(vehicle.rootNode, 0, 0, 1);
	local length = MathUtil.vector2Length(x,z);
	local dX = x/length
	local dZ = z/length
	local direction = 180 - math.deg(math.atan2(dX,dZ))
	FSContext.Telemetry.AngleRotation = direction;
end

function MS2Telemetry:ProcessMass(vehicle)
	if vehicle.getTotalMass ~= nil then
		FSContext.Telemetry.Mass = vehicle:getTotalMass(true);
		FSContext.Telemetry.TotalMass = vehicle:getTotalMass(false);
	else
		FSContext.Telemetry.Mass = 0.0;
		FSContext.Telemetry.TotalMass = 0.0;
	end
end

function MS2Telemetry:ProcessOnField(vehicle)
	FSContext.Telemetry.IsOnField = vehicle.getIsOnField ~= nil and vehicle:getIsOnField();
end

function MS2Telemetry:ProcessDef(vehicle)
	if vehicle.getConsumerFillUnitIndex ~= nil then
		local fillUnitIndex = vehicle:getConsumerFillUnitIndex(FillType.DEF);
		if fillUnitIndex ~= nil then
			FSContext.Telemetry.Def = vehicle:getFillUnitFillLevel(fillUnitIndex);
			FSContext.Telemetry.DefMax = vehicle:getFillUnitCapacity(fillUnitIndex);
			return;
		end
	end
	FSContext.Telemetry.DefMax = 0.0;
	FSContext.Telemetry.Def = 0.0;
end

function MS2Telemetry:ProcessAir(vehicle)
	if vehicle.getConsumerFillUnitIndex ~= nil then
		local fillUnitIndex = vehicle:getConsumerFillUnitIndex(FillType.AIR);
		if fillUnitIndex ~= nil then
			FSContext.Telemetry.Air = vehicle.getFillUnitFillLevel ~= nil and vehicle:getFillUnitFillLevel(fillUnitIndex) or 0.0;
			FSContext.Telemetry.AirMax = vehicle.getFillUnitFillLevel ~= nil and vehicle:getFillUnitCapacity(fillUnitIndex) or 0.0;
			return;
		end
	end
	FSContext.Telemetry.AirMax = 0.0;
	FSContext.Telemetry.Air = 0.0;
end

function MS2Telemetry:ProcessGameData()
	local player = MS2Telemetry:GetPlayer();
	if player ~= nil then
        local farm = g_farmManager:getFarmById(player.farmId)
		if farm ~= nil then
			FSContext.Telemetry.Money = farm.money;
		end

        MS2Telemetry:ProcessHudData(player);
	else
		self:Log("ProcessGameData - player is NIL");
    end

	if g_currentMission.environment ~= nil then
		local environment = g_currentMission.environment;
		if environment.weather ~= nil then
			local minTemp, maxTemp = environment.weather:getCurrentMinMaxTemperatures();
			FSContext.Telemetry.TemperatureMin = minTemp;
			FSContext.Telemetry.TemperatureMax = maxTemp;
			FSContext.Telemetry.TemperatureTrend = environment.weather:getCurrentTemperatureTrend();
		end

		FSContext.Telemetry.DayTimeMinutes = math.floor(environment.dayTime / (1000 * 60));

		local sixHours = 6 * 60 * 60 * 1000;
		local dayPlus6h, timePlus6h = environment:getDayAndDayTime(environment.dayTime + sixHours, environment.currentDay);
		FSContext.Telemetry.WeatherCurrent = environment.weather:getWeatherTypeAtTime(environment.currentDay, environment.dayTime);
		FSContext.Telemetry.WeatherNext = environment.weather:getWeatherTypeAtTime(dayPlus6h, timePlus6h);

		FSContext.Telemetry.Day = environment.currentDay;
	else
		self:Log("ProcessGameData - g_currentMission.environment is NIL");
	end

	-- Get the correct 'measuringUnit' as set by the user.
	local measuringUnit = 0; -- KPH
	if g_gameSettings:getValue("useMiles") then
		measuringUnit = 1; -- MPH
	end

	FSContext.Telemetry.SpeedUnit = measuringUnit;
end

function MS2Telemetry:Log(message)
	if self.LOG_ENABLED then
		print("MS2 Log: " .. message)
	end
end

function MS2Telemetry:ProcessGameEdition()
	self:Log("Processing game version: " .. tostring(g_minModDescVersion));
    if g_minModDescVersion >= 90 then
        FSContext.Telemetry.GameEdition = 25;
    elseif g_minModDescVersion >= 60 then
        FSContext.Telemetry.GameEdition = 22;
    else
        FSContext.Telemetry.GameEdition = 19;
    end
end

function MS2Telemetry:BuildHeaderText()
    self:BuildTelemetryKeys()

    local text = MS2Telemetry:AddText("HEADER", "")
    for _, key in ipairs(FSContext.TelemetryKeys) do
        text = MS2Telemetry:AddText(key, text)
    end
    return text
end

function MS2Telemetry:BuildBodyText()
    local text = MS2Telemetry:AddText("BODY", "")
    for _, key in ipairs(FSContext.TelemetryKeys) do
        local value = FSContext.Telemetry[key]
        text = MS2Telemetry:AddText(MS2Telemetry:GetTextValue(value), text)
    end
    return text
end

function MS2Telemetry:BuildTelemetryKeys()
	-- to ensure consistent order of headers/body pairs
    FSContext.TelemetryKeys = {}
    for key, _ in pairs(FSContext.Telemetry) do
        table.insert(FSContext.TelemetryKeys, key)
    end
    table.sort(FSContext.TelemetryKeys)
end

function MS2Telemetry:GetTextValue(value)
	local type = type(value);
	local text = "";
	if type == "boolean" then
		text = MS2Telemetry:GetTextBoolean(value);
	elseif type == "string" then
		text = value;
	elseif type =="number" then
		text = MS2Telemetry:GetTextDecimal(value);
	elseif type =="table" then
		text = MS2Telemetry:GetTextTable(value);
	end;
	return text;
end

function MS2Telemetry:GetTextDecimal(value)
	local integerPart, floatPart = math.modf(value);
	local numberText;
	if floatPart > 0 then
		numberText = string.format("%.2f", value);
	else
		numberText = string.format("%d", integerPart);
	end
	return numberText;
end

function MS2Telemetry:GetTextBoolean(value)
	return value and "1" or "0";
end

function MS2Telemetry:GetTextTable(valueTable)
	local text = "";
	for key, value in pairs(valueTable) do
		text = text .. MS2Telemetry:GetTextValue(value) .. "¶";
	end
	return text;
end

function MS2Telemetry:AddText(value, text)
	return text .. value .. "§";
end

function MS2Telemetry:WriteTelemetry()
    if FSContext.PipeControl.Pipe ~= nil then
        if FSContext.PipeControl.RefreshCurrent == 0 then
            FSContext.PipeControl.Pipe:write(MS2Telemetry:BuildHeaderText());
            FSContext.PipeControl.Pipe:flush();
        end
        FSContext.PipeControl.Pipe:write(MS2Telemetry:BuildBodyText());
        FSContext.PipeControl.Pipe:flush();
    else
        self:Log("Pipe is NIL in WriteTelemetry.");
    end
end


function MS2Telemetry:RefreshPipe()
	FSContext.PipeControl.RefreshCurrent = FSContext.PipeControl.RefreshCurrent + 1;
	if FSContext.PipeControl.RefreshCurrent >= FSContext.PipeControl.RefreshRate then
		FSContext.PipeControl.RefreshCurrent = 0;
	end

	if FSContext.PipeControl.RefreshCurrent == 0 then
		if FSContext.PipeControl.Pipe ~= nil then
			FSContext.PipeControl.Pipe:flush();
			FSContext.PipeControl.Pipe:close();
		end

		local pipe, err = io.open(FSContext.PipeControl.PipeName, "w");
        if pipe then
            FSContext.PipeControl.Pipe = pipe;
        else
            self:Log("Failed to open pipe: " .. tostring(err));
            FSContext.PipeControl.Pipe = nil;
        end
	end
end

addModEventListener(MS2Telemetry);
MS2Telemetry:Log("MS2Telemetry script loaded")
